--[[
[metadata]
description=This file provides a selection interface for previewing and selecting a new theme to load
]]--

local file_args = {...}

local public = file_args[1]
local private = file_args[2]
local config = file_args[3]

local he_ver = lib.get_latest("helium", "1.0.0", "1.3.9")

if (he_ver == "?") then
	private.cp("The interface module requires Helium (any version between 1.0.0 and 1.3.9) to be loaded! The CCD1 fallback will be provided instead.", 3)
	return
end

local helium = lib.get_class("helium", he_ver)



local create_help = function()
	local display = iup.stationsubframe {
		helium.primitives.solidframe {
		size = "%50x%50",
			iup.hbox {
				iup.fill {
					size = "%1",
				},
				iup.vbox {
					alignment = "ACENTER",
					iup.fill {
						size = "%1",
					},
					helium.constructs.shrink_label {
						title = private.bstr(9, "Reskin is a theme management mod for Vendetta Online. With reskin, skin packs can be accessed with a basic menu and applied easily. If you need a custom skin, you can download them from the game's wiki!"),
					},
					iup.stationbutton {
						title = "http://vo-wiki.com...",
						action = function()
							Game.OpenWebBrowser("http://vo-wiki.com/wiki/Skins")
						end,
					},
					iup.fill {
						size = "%2",
					},
					helium.constructs.shrink_label {
						title = private.bstr(10, "When using the interface, select an installed skin from the left-side list. If a thumbnail exists, it will appear, letting you preview what the skin will look like when applied. You can then use the apply button on the right bottom to save the skin to your game configuration. Afterwards, your game will reload."),
					},
					iup.fill {
						size = "%1",
					},
					helium.constructs.shrink_label {
						title = private.bstr(11, "Some parts of the game will not update after a reload. In order to change these element's visuals, you will need to fully close and reopen the game."),
					},
					iup.fill { },
					iup.stationbutton {
						title = private.bstr(12, "Okay"),
						action = function(self)
							HideDialog(iup.GetDialog(self))
						end,
					},
				},
				iup.fill {
					size = "%1",
				},
			},
		},
	}
	
	local diag = helium.presets.subdialog {
		display,
	}
	
	ShowDialog(diag)
end



local create_interface = function()
	--creates a simple list with preview
	--left list, right top preview & info, right bottom accept
	local cur_select = public.get_current()
	local cur_data = public.get_theme_data(cur_select)
	
	local thsize = tostring(gkinterface.GetYResolution() * 0.4)
	thsize = thsize .. "x" .. thsize
	local thumb_disp = iup.label {
		title = "",
		image = private.path .. "assets/missing_preview.png",
		--expand = "YES",
		size = thsize,
	}
	
	local thumb_info = iup.label {
		font = Font.Default + 4,
		title = "",
	}
	
	local thumb_container = iup.zbox {
		all = "YES",
		alignment = "ACENTER",
		iup.vbox {
			thumb_disp,
		},
		iup.hbox {
			iup.fill { },
			iup.vbox {
				thumb_info,
				iup.fill { },
			},
			iup.fill { },
		},
	}
	
	local update_thumb = function(sid)
		local data = public.get_theme_data(sid)
		if not data then
			return
		end
		cur_select = sid
		cur_data = data
		
		thumb_info.title = data.name
		thumb_disp.image = data.preview or private.path .. "assets/missing_preview.png"
		thumb_disp.size = thsize
		iup.Refresh(thumb_container)
	end
	
	local theme_list = iup.stationsublist {
		expand = "YES",
		size = "%10x%50",
		value = 1,
		action = function(self, t, i, cv)
			update_thumb(i)
		end,
	}
	
	do --update theme list on generation
		local theme_table = public.get_all()
		for index, name in ipairs(theme_table) do
			theme_list[index] = name
			if index == cur_select then
				theme_list.value = index
			end
		end
	end
	
	local display = iup.stationsubframe {
		helium.primitives.solidframe {
			size = "%50x%50",
			iup.vbox {
				iup.hbox {
					iup.label {
						title = private.bstr(1, "Reskin") .. " v" .. private.version,
					},
					iup.fill { },
					iup.stationbutton {
						title = private.bstr(14, "?"),
						action = function()
							create_help()
						end,
					},
					iup.stationbutton {
						title = private.bstr(15, "Close"),
						action = function(self)
							HideDialog(iup.GetDialog(self))
						end,
					},
				},
				iup.hbox {
					iup.vbox {
						theme_list,
					},
					iup.vbox {
						iup.hbox {
							iup.fill {
								size = "%2",
							},
							helium.constructs.shrink_label {
								title = private.bstr(13, "Changes to the game's theme will only occur after a reload. Some elements may need a full game restart."),
							},
							iup.fill {
								size = "%2",
							},
						},
						thumb_container,
						iup.stationbutton {
							title = private.bstr(8, "Apply and reload"),
							expand = "HORIZONTAL",
							action = function(self)
								local new_path = cur_data.dir
								gkini.WriteString("Vendetta", "skin", new_path)
								lib.reload()
							end,
						},
					},
				},
			},
		},
	}
	
	theme_list:action(nil, cur_select)
	
	local diag = helium.presets.subdialog {
		display,
	}
	
	ShowDialog(diag)
end

public.open = create_interface

RegisterUserCommand("reskin", public.open)