--[[
[metadata]
description=This file controls how the theme registry is used and accessed
]]--

local file_args = {...}

local public = file_args[1]
local private = file_args[2]
local config = file_args[3]


local currently_applied_skin = -1
local registry = {
	names_bookmark = {},
	dirs_bookmark = {},
	--[[
	[id] = {
		name = theme name
		dir = path/to/theme
		cfg = YES/NO has a config.ini for theme colors
		adv = YES/NO has a theme.lua for advance theme support
		preview = "thumbnail.png" or false
	]]--
}

local sanitize_index = function(index)
	if (not index) then
		return public.get_current()
	end
	
	if type(index) == "number" then
		return index or -1
	end
	
	if type(index) == "string" then
		if registry.names_bookmark[index] then
			--is index a skin name?
			return registry.names_bookmark[index]
		elseif registry.dirs_bookmark[index] then
			return registry.dirs_bookmark[index]
		else
			return tonumber(index) or -1
		end
	end
	
	return public.get_current()
end

public.register = function(skin_name, skin_dir)
	--add new skins to Reskin's registry
	if type(skin_name) ~= "string" or type(skin_dir) ~= "string" then
		return false, "incorrect argument format (expected two strings)"
	end
	
	if registry.dirs_bookmark[skin_dir] then
		return false, "skin already registered"
	end
	
	private.cp("Registering " .. skin_name .. " located at " .. skin_dir)
	
	local thumbnail = skin_dir .. "preview.png"
	
	local skin = {
		name = skin_name,
		dir = skin_dir,
		cfg = gksys.IsExist(skin_dir .. "config.ini") and "YES" or "NO",
		adv = gksys.IsExist(skin_dir .. "theme.lua") and "YES" or "NO",
		preview = gksys.IsExist(thumbnail) and thumbnail or false,
	}
	
	if skin.cfg == "YES" then
		lib.log_error("	skin provides a config file!")
	end
	if skin.adv == "YES" then
		lib.log_error("	skin contains an advanced theme generator!")
	end
	
	table.insert(registry, skin)
	registry.names_bookmark[skin_name] = #registry
	registry.dirs_bookmark[skin_dir] = #registry
	
	if skin.dir == IMAGE_DIR then
		currently_applied_skin = #registry
		--public.smart_config.skin_select_list.default = #themes
	end
	
	private.update_class() --add new entry to smart_config
	
	return true
end

public.add_data = function(index, input_table)
	--Add new data to an existing skin (cannot overwrite existing data)
	index = sanitize_index(index)
	
	local skin_table = registry[index]
	
	if not skin_table then
		return false, "not a valid index"
	end
	
	for k, v in pairs(input_table) do
		if (not skin_table[k]) then
			skin_table[k] = v
		end
	end
	
	return true
end

public.get_all = function()
	local ret = {}
	for k, v in ipairs(registry) do
		table.insert(ret, v.name)
	end
	
	return ret
end

public.get_current = function()
	return currently_applied_skin
end

public.set_current = function(index, reload_now)
	index = sanitize_index(index)
	if registry[index] then
		gkini.WriteString("Vendetta", "skin", registry[index].dir)
		if reload_now then
			lib.reload()
		end
	end
end

public.get_theme_data = function(skin_index)
	skin_index = sanitize_index(skin_index)
	
	if registry[skin_index] then
		return registry[skin_index]
	end
	
	return false, "no skin for provided index"
end

public.get_path = function(skin_index)
	skin_index = sanitize_index(skin_index)
	if registry[skin_index] then
		return registry[skin_index].dir
	else
		return false, "unknown skin"
	end
end

public.is_exist = function(index)
	return registry[sanitize_index(index)] and true or false
end