--[[
[modreg]
type=core
API=3
id=reskin
version=2.4.1
name=Reskin
author=Luxen
website=https://www.nexusmods.com/vendettaonline/mods/4
path=reskin.lua

[metadata]
description=This is the core file for Reskin, a theme management interface.
version=1.0.0
owner=Reskin|2.4.1
type=lua
created=2025-08-04
]]--

local re_path = lib.get_path() or "plugins/Reskin/"
local re_ver = lib.plugin_read_str(re_path .. "reskin.lua", nil, "modreg", "version")
local re_standalone = (re_path == "plugins/Reskin/") and "YES" or "NO"

local cp = function(msg, v)
	lib.log_error("[reskin] " .. tostring(msg), v or 1, "reskin", re_ver)
end

cp("Reskin " .. re_ver .. " is operating out of " .. re_path)



local config = {
	allow_advanced_theming = "YES",
}

local public = {
	CCD1 = true,
	commands = {},
	manifest = {
		"main.lua",
		"reskin.lua",
		
		"lang/en.ini",
		"lang/es.ini",
		"lang/fr.ini",
		"lang/pt.ini",
		
		"modules/interface.lua",
		"modules/registry.lua",
		"modules/helium/[helium v1.0.0]",
	},
	set_config = function(cfg, val)
		if config[cfg] then
			config[cfg] = tostring(val)
		end
	end,
	save_config = function()
		for k, v in pairs(config) do
			gkini.WriteString("reskin", k, v)
		end
	end,
	load_config = function()
		for k, v in pairs(config) do
			config[k] = gkini.ReadString("reskin", k, v)
		end
	end,
}

public.load_config()

local private
private = {
	path = re_path,
	version = re_ver,
	standalone = re_standalone,
	cp = cp,
	
	bstr = function(id, val)
		return val
	end,
	
	load_module = function(file_path, optional_flag)
		--optional_flag = false --uncomment for debugging interface.lua
		local valid_file_path = lib.find_file(private.path .. "modules/" .. file_path)
		if valid_file_path then
			cp("executing module " .. valid_file_path)
			
			local file_f, err = loadfile(valid_file_path)
			
			if file_f then
				file_f(public, private, config)
			elseif not optional_flag then
				error("failed to load required module " .. file_path .. ": " .. err)
			end
		elseif not optional_flag then
			error("failed to find required module " .. file_path)
		end
	end,
}

local update_class, babel, ref_id
local babel_func = function()
	babel = lib.get_class("babel", "0")
	ref_id = babel.register(re_path .. "lang/", {'en', 'es', 'fr', 'pt'})
	
	private.bstr = function(id, val)
		return babel.fetch(ref_id, id, val)
	end
	
	public.add_translation = function(path, lang_code)
		babel.add_new_lang(ref_id, path, lang_code)
	end
	
	update_class()
end



local smart_config_list_select_option = ""

update_class = function()
	local class = {
		description = private.bstr(2, "Reskin is a theme selection interface. Add themes to the game and switch between them easily with this utility!"),
	}
	
	class.smart_config = {
		title = private.bstr(1, "Reskin"),
		cb = function(id, val)
			if id == "update_check" then
				Game.OpenWebBrowser("https://www.nexusmods.com/vendettaonline/mods/4")
				return
			elseif id == "wikilink" then
				Game.OpenWebBrowser("http://vo-wiki.com/wiki/Skins")
				return
			elseif id == "apply_now" then
				public.set_current(smart_config_list_select_option, true)
			elseif id == "skin_select_list" then
				smart_config_list_select_option = val
			else
				if config[id] then
					config[id] = tostring(val)
				end
			end
		end,
		"update_check",
		"spacer",
		"theme_text",
		"wikilink",
		"skin_select_list",
		"spacer",
		"adv_theme_text",
		"allow_advanced_theming",
		"spacer",
		
		update_check = {
			type = "action",
			display = "",
			align = "right",
			[1] = private.bstr(3, "Check for updates") .. "...",
		},
		spacer = {
			type = "spacer",
		},
		theme_text = {
			type = "text",
			display = private.bstr(4, "You can select a theme from the dropdown below. Reload the game to see changes. Some files may need a full restart instead."),
			align = "center",
		},
		wikilink = {
			type = "action",
			display = private.bstr(5, "You can download new themes from the game's wiki"),
			align = "right",
			[1] = "http://vo-wiki.com...",
		},
		skin_select_list = {
			type = "dropdown",
			display = "",
			default = 1,
		},
		
		adv_theme_text = {
			type = "text",
			display = private.bstr(6, "Some skins may support advanced theme capabilities."),
			align = "left",
		},
		allow_advanced_theming = {
			type = "toggle",
			display = private.bstr(7, "Allow advanced theme support?"),
			[1] = config.allow_advanced_theming,
		},
	}
	
	--for each skin, add to sc
	--set default to 'current skin'
	local select_list = class.smart_config.skin_select_list
	
	local theme_list = public.get_all()
	for index, theme in ipairs(theme_list) do
		--cp("adding option to smart config list: " .. theme)
		select_list[index] = theme
	end
	
	select_list.default = public.get_current()
	
	smart_config_list_select_option = select_list[select_list.default]
	
	table.insert(class.smart_config, "apply_now")
	class.smart_config.apply_now = {
		type = "action",
		display = "",
		align = "right",
		[1] = private.bstr(8, "Apply and reload"),
	}
	
	for k, v in pairs(class) do
		public[k] = v
	end
	
	lib.set_class("reskin", private.version, public)
end

private.update_class = update_class



private.load_module("registry.lua")

lib.require({{name = "helium", version = "1.0.0", ver_max = "1.3.9"}}, function()
	private.load_module("interface.lua", true)
end)

update_class()

lib.require({{name = "babel", version = "0"}}, babel_func)





local apply_theme = function()
	--this runs at PLUGINS_LOADED and executes the theme.lua file if a skin has one. This will let skins provide even more customization that strict INI/image manipulation cannot. Only the active skin will have its theme.lua file executed.
	
	if config.allow_advanced_theming ~= "YES" then
		return
	end
	
	local skin_table = public.get_theme_data()
	if not skin_table then
		cp("Reskin couldn't determine the current skin! The current skin is " .. IMAGE_DIR .. "; verify the skin in this folder supports Reskin.")
		return
	end
	
	if skin_table.adv == "YES" then
		cp("The current skin has an advanced theme file. This file will be executed now.", 3)
		lib.resolve_file(skin_table.dir .. "theme.lua")
	end
end

RegisterEvent(apply_theme, "PLUGINS_LOADED")



public.register("Themeless", "skins/invalid/")
public.register("Default Platinum", "skins/platinum/")
public.register("Default Blue", "images/station/")
--public.add_data("Themeless", {preview = re_path .. "assets/themeless.png"})
public.add_data("Default Platinum", {preview = re_path .. "assets/platinum_preview.png"})
public.add_data("Default Blue", {preview = re_path .. "assets/blue_preview.png"})